<?php  if (!defined('BASEPATH')) exit('No direct script access allowed');

/**
 * MODULE NAME   : events.php
 *
 * DESCRIPTION   : events module controller
 *
 * MODIFICATION HISTORY
 *   V1.0   2008-10-31 11:59 PM   - Leane Verhulst     - Created
 *
 * @package			ConCentric
 * @subpackage		events component Class
 * @author			Leane Verhulst
 * @copyright		Copyright (c) 2008
 * @license			http://www.gnu.org/licenses/gpl.html
 */

class Events extends Concentric_Controller
{
	/**
	* Contructor function
	*
	* Load the instance of CI by invoking the parent constructor
	*
	* @access      public
	* @return      none
	*/
	function Events()
	{
		// Call parent constructor
		parent::Concentric_Controller();

		$this->form_name = "Events";
		$this->load->model('events_model');    // Instantiate the model
		// load helper library
		$this->load->helper('concentric');
	}

	// -------------------------------------------------------------------------
	/**
	* "Index" Page
	*
	* Default class action
	*
	* @access      public
	* @return      none
	*/
	function index($sortby = '')
	{
		// The default action is the browse action
		$this->browse($sortby);
	}

	// -------------------------------------------------------------------------
	// Function: browse()
	// Description: Extracts a list of all data records and
	//              displays it.
	function browse($sortby = '')
	{
		// get sort from cookie / save new sort to cookie
		$array = get_events_sort($sortby);
		$sortname = $array['name'];
		$sortdirection = $array['direction'];
		$sortby = $array['sortby'];

		// fetch query object from the model
		$template['events_list'] = $this->events_model->fetchEvents(NULL, $sortby);

		// create urls for use in content view
		$template['base_url']   = 'concentric/events/browse/';
		$template['edit_url']   = 'concentric/events/edit/';
		$template['delete_url'] = 'concentric/events/delete/';
		$template['add_url']    = 'concentric/events/add/';
		$template['cancel_url'] = 'concentric/home';
		$template['sort_url']['Name']      = 'concentric/events/browse/name';
		$template['sort_url']['Category']  = 'concentric/events/browse/category';
		$template['sort_url']['Track']     = 'concentric/events/browse/track';
		$template['sort_url']['Date']      = 'concentric/events/browse/date';
		$template['sort_url']['Active']    = 'concentric/events/browse/active';
		$template['sort_url']['Hide']      = 'concentric/events/browse/hide';
		$template['sort_url']['Public']    = 'concentric/events/browse/public';
		$template['sort']['name']          = $sortname;
		$template['sort']['direction']     = $sortdirection;

		// Set breadcrumb
		$this->page->set_crumb('Events',$template['base_url']);

		// set values for use in content view
		$template['form_attributes'] = array('name' => $this->form_name, 'id' => $this->form_name);
		$template['header'] = $this->form_name;

		// build content html using grid view and template data
		$data['content'] = $this->load->view('/concentric/events/events_grid', $template, TRUE);

		// set values for use in header view
		$data['header']  = $this->form_name;

		// pass all data to container view and display to user
		$this->load->view($this->_container,$data);
	}

	// -------------------------------------------------------------------------
	// Function: add()
	// Description: Prompts user for input and adds a new entry
	//              in the database.
	function add()
	{
		$this->_modify('add');
	}

	// -------------------------------------------------------------------------
	// Function: edit()
	// Description: Prompts user for input and updates an entry
	//              in the database.
	function edit()
	{
		$this->_modify('edit');
	}

	// -------------------------------------------------------------------------
	// Function: modify()
	// Description: Prompts user for input and adds/edits an entry
	//              in the database. Default to add mode.
	function _modify($mode = 'add')
	{
		// Initialize variable
		$other_scripts_data = '';

		// load validation library to help validate data
		$this->load->library('validation');

		// load grid helper
		$this->load->helper('grid');

		// Set breadcrumb
		$this->page->set_crumb('Events','concentric/events/');
		$this->page->set_crumb('Details','concentric/events/' . $mode);

		// initialize date arrays
		$this->dates_array = datetime_array($this->ccsettings->item('start_date_time'), $this->ccsettings->item('end_date_time'), $this->ccsettings->item('interval_length'));
		$this->dates_array_flip = array_flip($this->dates_array);

		// create urls and values for use in content view
		$template['form_url'] = 'concentric/events/' . $mode . '/';
		$template['cancel_url'] = 'concentric/events/browse';
		$template['form_attributes'] = array( 'method' => 'post', 'id' => $this->form_name, 'name' => $this->form_name );
		$template['header']  = $this->form_name;
		//if ($mode == 'add')
		//{
			$template['submit_button_name']  = 'Save';
		//} else {
		//	$template['submit_button_name']  = 'Save and Next';
		//}

		// create html to use the editor
		$editor_url = $this->config->item('base_url') . 'assets/tiny_mce/';
		$other_scripts_data .= $this->_tinymce_scripts($editor_url);

		// create html to use the calendar picker
		$this->cal_url = $this->config->item('base_url') . 'assets/tigra_calendar/';
		$other_scripts_data .= $this->_calendar_script($this->cal_url);


		$this->_initialize();   // Setup the fields
		$this->_validation();   // Load the validation rules and fields


		// if validation has not run or if it failed
		if ( $this->validation->run() == FALSE )
		{
			// Save validation errors to flashmsg
			// Will be output in content view by status->display
			$this->validation->output_errors();

			// Get values for form fields and build the form fields
			$status = $this->_get_form_values($mode);
			// If the returned values got an error
			if ($status === FALSE)
			{
				// Show error message
				flashMsg('error','There was a problem editing that record.');
				// redirect to main location
				redirect('/concentric/events/', 'location');
			}

			// Save field data to array to be passed to content view
			$template['fields'] = $this->fields;


			// if editing an event, create html string of additional details - locations and participants
			$template['additional_details'] = '';
			if ($mode != 'add')
			{
				$template['additional_details'] = $this->_get_additional_details();
			}


			// build content html using details view and template data
			$data['content'] = $this->load->view('/concentric/details', $template, TRUE);
			// pass other scripts data to container
			$data['other_scripts'] = $other_scripts_data;

			// pass all data to container view and display to user
			$this->load->view($this->_container, $data);
		}
		else  // User is submitting valid data
		{
			// Store the values from the form into the db
			// XXS Filtering enforced for user input
			$data['field']['id']				= $this->input->post('id', TRUE);
			$data['field']['name']				= $this->input->post('name', TRUE);
			$data['field']['startDateTime']		= $this->input->post('startDateTime', TRUE);
			$data['field']['startDateTime']		= $this->dates_array[$data['field']['startDateTime']];
			$data['field']['description']		= $this->input->post('description', TRUE);
			$data['field']['eventLength']		= $this->input->post('eventLength', TRUE);
			$data['field']['setupTime']			= $this->input->post('setupTime', TRUE);
			$data['field']['teardownTime']		= $this->input->post('teardownTime', TRUE);
			$data['field']['reimbursibleQ']		= $this->input->post('reimbursibleQ', TRUE);
			$data['field']['activeQ']			= $this->input->post('activeQ', TRUE);
			$data['field']['locationSetupID']	= $this->input->post('locationSetupID', TRUE);
			$data['field']['categoryID']		= $this->input->post('categoryID', TRUE);
			$data['field']['hideQ']				= $this->input->post('hideQ', TRUE);
			$data['field']['hostDescription']	= $this->input->post('hostDescription', TRUE);
			$data['field']['hidePanelistsQ']	= $this->input->post('hidePanelistsQ', TRUE);
			$data['field']['trackID']			= $this->input->post('trackID', TRUE);

			if ($mode == 'add')
			{
				// add the data to the table and return id
				$idField = $this->events_model->addRow('events',$data['field']);
				$status = ($idField) ? TRUE : FALSE;
			}
			else
			{
				$idField = $data['field']['id'];

				// update the data in the table
				$obj = $this->events_model->update('events', $data['field'], 'id = ' . $idField);
				$status = ($obj) ? TRUE : FALSE;
			}

			// if there was no error in add or edit
			if ($status)
			{
				// Show success message
				flashMsg('success','Record ' . $idField . ' (' . $data['field']['name'] . ') has been saved.');
			}
			else  // there was an error
			{
				// Show error message
				flashMsg('error','The record for (' . $data['field']['name'] . ') has not been saved.');
			}

			/*
			// if editing, get next id based on sorted name and redirect back to edit
			if ($mode == 'edit')
			{
				$nextId = $this->events_model->getNextByName($data['field']['name']);

				// if no more ids, then go back to main location
				if ($nextId == 0)
				{
					redirect('/concentric/events/', 'location');
				}

				redirect('/concentric/events/edit/' . $nextId, 'location');
			}
			*/

			// redirect to main location
			redirect('/concentric/events/', 'location');
			// ?redirect to add/edit?
			// redirect('/concentric/events/' . $mode . '/', 'location');
		}
	}

	// -------------------------------------------------------------------------
	// Function: delete()
	// Description: Controller function to process user delete requests
	function delete()
	{
		$idField = $this->uri->segment(4);

		//$confirm = $this->_delete_confirm($idField);  //add later
		$confirm = TRUE;

		if ($confirm)
		{
			if ( $this->events_model->delete('events', 'id =' . $idField) )
			{
				flashMsg('success', 'The record has been deleted.');
			}
			else
			{
				flashMsg('error', 'There was a problem deleting the record');
			}
		}

		redirect('/concentric/events/', 'location');
	}

	// -------------------------------------------------------------------------
	function _delete_confirm($id)
	{
		//add later
	}

	// -------------------------------------------------------------------------
	// @access private
	function _get_form_values($mode = 'add')
	{
		if ($mode == 'add')
		{
			// Set empty values / default values
			$this->fields['id']['value'] = '';
			$this->fields['name']['value'] = '';
			$this->fields['startDateTime']['value'] = 0;
			$this->fields['description']['value'] = '';
			$this->fields['eventLength']['value'] = 0;
			$this->fields['setupTime']['value'] = 0;
			$this->fields['teardownTime']['value'] = 0;
			$this->fields['hostDescription']['value'] = 'Panelists';
			$this->fields['locationSetupID']['value'] = '';
			$this->fields['locationSetupID']['selected'] = 1;
			$this->fields['categoryID']['value'] = '';
			$this->fields['categoryID']['selected'] = 1;
			$this->fields['reimbursibleQ']['value'] = 1;
			$this->fields['reimbursibleQ']['checked'] = TRUE;
			$this->fields['activeQ']['value'] = 1;
			$this->fields['activeQ']['checked'] = TRUE;
			$this->fields['hideQ']['value'] = 1;
			$this->fields['hideQ']['checked'] = FALSE;
			$this->fields['hidePanelistsQ']['value'] = 1;
			$this->fields['hidePanelistsQ']['checked'] = FALSE;
			$this->fields['trackID']['value'] = '';
			$this->fields['trackID']['selected'] = 1;
		}
		else
		{
			//get the id from either the hidden field or the url
			$idField = ($this->input->post('id', TRUE) != '') ? $this->input->post('id', TRUE) : $this->uri->segment(4);

			// fetch query object from the model with all fields
			$qry = $this->events_model->fetch('events','id, name, startDateTime, description, eventLength, setupTime, teardownTime, hostDescription, locationSetupID, categoryID, reimbursibleQ, activeQ, hideQ, hidePanelistsQ, trackID','','id = ' . $idField);

			// There should only be 1 row. If not, return FALSE
			if ($qry->num_rows() != 1)
			{
				return FALSE;
			}

			// Get the single result as an array and save values
			$row = $qry->row_array();
			$this->fields['id']['value'] = $row['id'];
			$this->fields['name']['value'] = $row['name'];
			$this->fields['startDateTime']['value'] = ($row['startDateTime'] == '0000-00-00 00:00:00') ? '' : $row['startDateTime'];
			$this->fields['startDateTime']['selected'] = ($this->fields['startDateTime']['value'] == '') ? '' : $this->dates_array_flip[$this->fields['startDateTime']['value']];
			$this->fields['description']['value'] = $row['description'];
			$this->fields['eventLength']['value'] = $row['eventLength'];
			$this->fields['setupTime']['value'] = $row['setupTime'];
			$this->fields['teardownTime']['value'] = $row['teardownTime'];
			$this->fields['hostDescription']['value'] = $row['hostDescription'];
			$this->fields['locationSetupID']['value'] = $row['locationSetupID'];
			$this->fields['locationSetupID']['selected'] = $row['locationSetupID'];
			$this->fields['categoryID']['value'] = $row['categoryID'];
			$this->fields['categoryID']['selected'] = $row['categoryID'];
			$this->fields['reimbursibleQ']['value'] = 1;
			$this->fields['reimbursibleQ']['checked'] = ($row['reimbursibleQ']) ? TRUE : FALSE;
			$this->fields['activeQ']['value'] = 1;
			$this->fields['activeQ']['checked'] = ($row['activeQ']) ? TRUE : FALSE;
			$this->fields['hideQ']['value'] = 1;
			$this->fields['hideQ']['checked'] = ($row['hideQ']) ? TRUE : FALSE;
			$this->fields['hidePanelistsQ']['value'] = 1;
			$this->fields['hidePanelistsQ']['checked'] = ($row['hidePanelistsQ']) ? TRUE : FALSE;
			$this->fields['trackID']['value'] = $row['trackID'];
			$this->fields['trackID']['selected'] = $row['trackID'];
		}

		// Now build the form fields
		foreach ($this->fields as $field)
		{
			$this->fields[$field['name']]['form'] = build_form_field($field);
		}

		return TRUE;
	}


	// -------------------------------------------------------------------------
	// @access private
	function _get_additional_details()
	{
		// need these models to get details
		$this->load->model('eventlocations_model');    // Instantiate the model
		$this->load->model('eventparticipants_model');    // Instantiate the model

		//get the id from either the hidden field or the url
		$idField = ($this->input->post('id', TRUE) != '') ? $this->input->post('id', TRUE) : $this->uri->segment(4);

		$eventlocations_list = $this->eventlocations_model->fetchEventLocations('event.id = '.$idField);
		$eventparticipants_list = $this->eventparticipants_model->fetchEventParticipants('event.id = '.$idField);

		//loop thru query results and build string
		foreach ($eventlocations_list->result_array() as $row)
		{
			$locationarr[] .= $row['location_name'];
		}
		$locationdata = '';
		if (isset($locationarr))
			$locationdata = '<p>Location: ' . implode(', ', $locationarr) . '</p>';

		//loop thru query results and build string
		foreach ($eventparticipants_list->result_array() as $row)
		{
			$partarr[] .= $row['participant_name'];
		}
		$partdata = '';
		if (isset($partarr))
			$partdata = '<p>Participants: ' . implode(', ', $partarr) . '</p>';


		return $locationdata . $partdata;
	}


	// -------------------------------------------------------------------------
	// @access private
	function _validation()
	{
		// load the rules
		$form_rules['name']  = $this->fields['name']['rules'];
		$form_rules['startDateTime']  = $this->fields['startDateTime']['rules'];
		$form_rules['eventLength']  = $this->fields['eventLength']['rules'];
		$form_rules['setupTime']  = $this->fields['setupTime']['rules'];
		$form_rules['teardownTime']  = $this->fields['teardownTime']['rules'];
		$form_rules['hostDescription']  = $this->fields['hostDescription']['rules'];
		$form_rules['locationSetupID']  = $this->fields['locationSetupID']['rules'];
		$form_rules['categoryID']  = $this->fields['categoryID']['rules'];
		$form_rules['reimbursibleQ']  = $this->fields['reimbursibleQ']['rules'];
		$form_rules['activeQ']  = $this->fields['activeQ']['rules'];
		$form_rules['hideQ']  = $this->fields['hideQ']['rules'];
		$form_rules['hidePanelistsQ']  = $this->fields['hidePanelistsQ']['rules'];
		$form_rules['trackID']  = $this->fields['trackID']['rules'];
		$this->validation->set_rules($form_rules);

		// load the field labels
		$form_fields['id']   = $this->fields['id']['label'];
		$form_fields['name']  = $this->fields['name']['label'];
		$form_fields['startDateTime']   = $this->fields['startDateTime']['label'];
		$form_fields['description']   = $this->fields['description']['label'];
		$form_fields['eventLength']   = $this->fields['eventLength']['label'];
		$form_fields['setupTime']   = $this->fields['setupTime']['label'];
		$form_fields['teardownTime']   = $this->fields['teardownTime']['label'];
		$form_fields['locationSetupID']   = $this->fields['locationSetupID']['label'];
		$form_fields['categoryID']   = $this->fields['categoryID']['label'];
		$form_fields['reimbursibleQ']   = $this->fields['reimbursibleQ']['label'];
		$form_fields['activeQ']   = $this->fields['activeQ']['label'];
		$form_fields['hideQ']   = $this->fields['hideQ']['label'];
		$form_fields['hidePanelistsQ']   = $this->fields['hidePanelistsQ']['label'];
		$form_fields['trackID']   = $this->fields['trackID']['label'];
		$this->validation->set_fields($form_fields);

		// set error delimiters
		$this->validation->set_error_delimiters('<span class="error">', '</span>');

	}


	// -------------------------------------------------------------------------
	/**
	 * Initalize Values
	 * 
	 * @access private
	 * @return void;
	 */
	function _initialize()
	{
		// Setup custom field options - list fields in the order they should
		// appear in the details view
		//   rules - validation rules (optional)
		//   label - descriptive name of the field (optional)
		//   name - database field name ** MUST BE SAME AS KEY ** (required)
		//   desc - long description to show to user on form (optional)
		//   type - set of values (required)
		//            text - basic input box
		//            hidden - keep this field hidden
		//            textarea
		//            checkbox
		//            dropdown
		//            password
		//   class - class name (optional)
		//   script - html script code to insert on the field
		//   after_form_script - html script code to insert after the form has been displayed

		$this->fields['id'] = array('label' => 'Event ID',
							'name' => 'id',
							'type' => 'hidden');
		$this->fields['name'] = array('rules' => 'trim|required|max_length[75]|callback__unique_field[events.name]|xss_clean',
							'label' => 'Event Name',
							'name' => 'name',
							'desc' => 'Enter the name of the event.',
							'type' => 'text');
		$this->fields['startDateTime'] = array('rules' => 'trim|callback__check_location_conflict|callback__check_part_conflict|xss_clean',
							'label' => 'Start Date/Time',
							'name' => 'startDateTime',
							'desc' => 'Enter the start date and time for the event in yyyy-mm-dd hh:mi:ss format. Valid dates are from ' . $this->ccsettings->item('start_date_time') . ' thru ' . $this->ccsettings->item('end_date_time'),
							'type' => 'dropdown',
							'options' => $this->dates_array,
							'selected' => '');
		$this->fields['eventLength'] = array('rules' => 'trim|max_length[5]|numeric|callback__check_interval|xss_clean',
							'label' => 'Event Length',
							'name' => 'eventLength',
							'desc' => 'Enter the length of the event as hours and fractions of hours. A half-hour event would be .5. This must be evenly divisible by the interval length.',
							'type' => 'text');
		$this->fields['setupTime'] = array('rules' => 'trim|max_length[5]|numeric|callback__check_interval|xss_clean',
							'label' => 'Setup Time',
							'name' => 'setupTime',
							'desc' => 'Enter the length of the time needed for setup as hours and fractions of hours. When checking for conflicts, other events will not be able to overlap this time. This must be evenly divisible by the interval length.',
							'type' => 'text');
		$this->fields['teardownTime'] = array('rules' => 'trim|max_length[5]|numeric|callback__check_interval|xss_clean',
							'label' => 'Teardown Time',
							'name' => 'teardownTime',
							'desc' => 'Enter the length of the time needed for teardown as hours and fractions of hours. When checking for conflicts, other events will not be able to overlap this time. This must be evenly divisible by the interval length.',
							'type' => 'text');
		$this->fields['hostDescription'] = array('rules' => 'trim|max_length[50]|alpha_dash_space|xss_clean',
							'label' => 'Description for Panelists',
							'name' => 'hostDescription',
							'desc' => 'Enter the title to use for the panelists of the event. By default, it will be Panelists.  Other options: Game Masters, Presenters, etc.',
							'type' => 'text');
		$this->fields['locationSetupID'] = array('rules' => 'trim|xss_clean',
							'label' => 'Location Setup',
							'name' => 'locationSetupID',
							'desc' => 'Enter the location setup option to use for this event.',
							'type' => 'dropdown',
							'options' => $this->_get_locationSetup_array(),
							'selected' => '');
		$this->fields['categoryID'] = array('rules' => 'trim|xss_clean',
							'label' => 'Reporting Category',
							'name' => 'categoryID',
							'desc' => 'Enter the reporting category for the event.',
							'type' => 'dropdown',
							'options' => $this->_get_category_array(),
							'selected' => '');
		$this->fields['trackID'] = array('rules' => 'trim|xss_clean',
							'label' => 'Track',
							'name' => 'trackID',
							'desc' => 'Enter the track type for the event.',
							'type' => 'dropdown',
							'options' => $this->_get_track_array(),
							'selected' => '');
		$this->fields['reimbursibleQ'] = array('rules' => 'trim|xss_clean',
							'label' => 'Reimbursible?',
							'name' => 'reimbursibleQ',
							'desc' => 'Is this event a reimbursible event? Events that are not reimbursible are not counted when calculating a panelists reimbursement time.',
							'type' => 'checkbox');
		$this->fields['activeQ'] = array('rules' => 'trim|xss_clean',
							'label' => 'Active?',
							'name' => 'activeQ',
							'desc' => 'Is this event active? This option can be used to suppress the event from displaying on most reports without having to delete it.',
							'type' => 'checkbox');
		$this->fields['hideQ'] = array('rules' => 'trim|xss_clean',
							'label' => 'Hide?',
							'name' => 'hideQ',
							'desc' => 'Is this event an event that should be hidden on public reports? This can be used to suppress events, such as private board meetings, which still need to be scheduled around, but which should not show up on public reports.',
							'type' => 'checkbox');
		$this->fields['hidePanelistsQ'] = array('rules' => 'trim|xss_clean',
							'label' => 'Hide Panelists?',
							'name' => 'hidePanelistsQ',
							'desc' => 'Should panelists be hidden for this event on public reports? Certain events, such as Opening Ceremonies, need to have people scheduled, but these people do not need to be displayed on public reports.',
							'type' => 'checkbox');
		$this->fields['description'] = array('label' => 'Description',
							'name' => 'description',
							'desc' => 'Enter a full description for the event as it should appear in the program book.',
							'type' => 'textarea',
							'class' => 'mceEditor');

		return;
	}

	// -------------------------------------------------------------------------
	/**
	 * Unique_field
	 * ** Assumes that the db model has already been loaded. **
	 *
	 * @access    private
	 * @param     string  The string to check for in the db.
	 * @param     field   The table and column of the field to check in table.column format
	 * @return    bool
	 */
	function _unique_field($str, $field)
	{
		list($table, $column) = split("\.", $field, 2);

		// Set the error message to use
		$this->validation->set_message('_unique_field','The %s field must have a unique value in the database.');

		$where= isset($_POST['id']) ? array('id !='=> $_POST['id'], $column=>$str) : array($column=>$str);
		$count = $this->events_model->count_condition($table, $where);
		return ($count > 0) ? FALSE : TRUE;
	}

	// -------------------------------------------------------------------------
	/**
	 * Get locationSetup array
	 *
	 * @access    private
	 * @return    array
	 */
	 function _get_locationSetup_array()
	{
		// Load the model
		$this->load->model('locationsetups_model');

		// fetch query object from the model with fields id, name
		$query = $this->locationsetups_model->fetch('locationSetups','id, name','','','name ASC');

		// Loop through to build array
		$data = array();
		foreach ($query->result_array() as $row)
		{
			$data[$row['id']] = $row['name'];
		}

		return $data;
	}

	// -------------------------------------------------------------------------
	/**
	 * Get category array
	 *
	 * @access    private
	 * @return    array
	 */
	 function _get_category_array()
	{
		// Load the model
		$this->load->model('categories_model');

		// fetch query object from the model with fields id, name
		$query = $this->categories_model->fetch('categories','id, name','','','name ASC');

		// Loop through to build array
		$data = array();
		foreach ($query->result_array() as $row)
		{
			$data[$row['id']] = $row['name'];
		}

		return $data;
	}

	// -------------------------------------------------------------------------
	/**
	 * Get track array
	 *
	 * @access    private
	 * @return    array
	 */
	 function _get_track_array()
	{
		// Load the model
		$this->load->model('tracks_model');

		// fetch query object from the model with fields id, name
		$query = $this->tracks_model->fetch('tracks','id, name','','','name ASC');

		// Loop through to build array
		$data = array();
		foreach ($query->result_array() as $row)
		{
			$data[$row['id']] = $row['name'];
		}

		return $data;
	}

	// -------------------------------------------------------------------------
	//return html for tiny_mce editor script info
	// @access private
	// @return string html
	function _tinymce_scripts($url = '')
	{
		$str = "\n";
		$str .= '<script type="text/javascript" src="' . $url . 'tiny_mce.js"></script>' . "\n";
		$str .= '<script type="text/javascript">' . "\n";
		$str .= 'tinyMCE.init({' . "\n";
		$str .= '	mode : "textareas",' . "\n";
		$str .= '	theme : "advanced",' . "\n";
		$str .= '	editor_selector : "mceEditor",' . "\n";
		$str .= '	editor_deselector : "mceNoEditor",' . "\n";
		$str .= '	theme_advanced_buttons1 : "bold,italic,underline,|,bullist,numlist,|,undo,redo,cleanup,help",' . "\n";
		$str .= '	theme_advanced_buttons2 : "fontselect,fontsizeselect,|,blockquote,hr,code",' . "\n";
		$str .= '	theme_advanced_buttons3 : "",' . "\n";
		$str .= '	theme_advanced_toolbar_location : "top",' . "\n";
		$str .= '	theme_advanced_toolbar_align : "left",' . "\n";
		$str .= '	theme_advanced_statusbar_location : "bottom",' . "\n";
		$str .= '	theme_advanced_resizing : true' . "\n";
		$str .= '});' . "\n";
		$str .= '</script>' . "\n" . "\n";

		return $str;
	}

	// -------------------------------------------------------------------------
	// @access private
	// @return string html
	function _calendar_script($url = '')
	{
		$str = "\n";
		$str .= '<script language="JavaScript" src="' . $url . 'calendar3.js"></script>' . "\n";

		return $str;
	}

	// -------------------------------------------------------------------------
	// @access private
	// @return string html
	function _calendar_picker($element = 'cal', $name = '', $url = '')
	{
		$str = "\n";
		$str .= '<script language="JavaScript">' . "\n";
		$str .= '	var cal_' . $element . ' = new calendar3(document.forms["' . $name . '"].elements["' . $element . '"]);' . "\n";
		$str .= '	cal_' . $element . '.year_scroll = true;' . "\n";
		$str .= '	cal_' . $element . '.time_comp = true;' . "\n";
		$str .= '	cal_' . $element . '.location = "' . $url . '";' . "\n";
		$str .= '</script>' . "\n";

		return $str;
	}

	// -------------------------------------------------------------------------
	// @access private
	// @return string html
	function _calendar_link($element = 'cal', $url = '')
	{
		//$sdttm = $this->ccsettings->item('start_date_time');
		//$str = '';
		$str = '<a href="javascript:cal_' . $element . '.popup();"><img src="' . $url . 'img/cal.gif" width="16" height="16" border="0" alt="Click here to pick the date"></a>';
		return $str;
	}

	// -------------------------------------------------------------------------
	/**
	 * Check that the time entered is evenly divisible by the interval length
	 *
	 * @access    private
	 * @param     string  The string to check.
	 * @return    bool
	 */
	function _check_interval($str)
	{
		// Set the error message to use
		$this->validation->set_message('_check_interval','The %s field must be evenly divisible by the interval length in settings.');

		// get interval length
		$intval = $this->ccsettings->item('interval_length');

		// convert time entered to seconds, divide by interval, get remainder 
		$data = ($str * 3600) % $intval;

		// if no remainder, good, else bad
		return ($data == 0) ? TRUE : FALSE ;
	}

	// -------------------------------------------------------------------------
	/**
	 * Check that event does not have a conflict before scheduling
	 *
	 * @access    private
	 * @param     string  The string to check.
	 * @return    bool
	 */
	function _check_location_conflict($str)
	{
		// Set the error message to use
		$this->validation->set_message('_check_location_conflict','The %s field cannot be scheduled due to a conflict.');

		// if adding a new event, there can be no conflicts as no location or participant is assigned
		$id = $this->input->post('id', TRUE);
		if ($id == '')
			return TRUE;  //good, no conflicts

		$str = $this->dates_array[$str];  //get date string

		// if date is blank, there can be no conflicts
		if ($str == '0000-00-00 00:00:00' OR $str == '')
			return TRUE;  //good, no conflicts

		// get locations used by this event
		$this->load->model('eventlocations_model');
		$query = $this->eventlocations_model->fetchEventLocations('event.id = '.$id);
		if ($query->num_rows() == 0)
			return TRUE;  //good, no conflicts
		// convert locations query to array
		foreach ($query->result_array() as $row)
		{
			$locationsort[$row['location_sortorder']] = $row['location_id'];  //location sortOrder
		}
		$query->free_result();


		// Need to create a list of locations for grid
		$this->load->model('locations_model');    // Instantiate the model
		$query = $this->locations_model->fetch('locations', 'id, sortOrder', '', '', 'sortOrder ASC');  //get data
		foreach ($query->result_array() as $row)
		{
			$locations[$row['sortOrder']] = $row['id'];
		}
		$locationGrid = $this->events_model->build_location_grid($locations);  // Build the location grid for conflict checking
		$query->free_result();


		// get event data
		$sdttm = strtotime($str);
		$edttm = strtotime("+".($this->input->post('eventLength', TRUE)*3600).' seconds', $sdttm);
		$setup = strtotime("-".($this->input->post('setupTime', TRUE)*3600).' seconds', $sdttm);
		$teardown = strtotime("+".($this->input->post('teardownTime', TRUE)*3600).' seconds', $edttm);
		$eventInfo = array('id' => $id,
							'sdttm' => date('Y-m-d H:i:s',$sdttm),
							'edttm' => date('Y-m-d H:i:s',$edttm),
							'setupdttm' => date('Y-m-d H:i:s',$setup),
							'teardowndttm' => date('Y-m-d H:i:s',$teardown),
							'setup' => $this->input->post('setupTime', TRUE),
							'teardown' => $this->input->post('teardownTime', TRUE),
							'eventlength' => $this->input->post('eventLength', TRUE)
							);

		// check that an event can fit on grid and return status
		$status = check_event_on_locationgrid($locationGrid, $locationsort, $eventInfo, $this->ccsettings->item('interval_length'));

		// if status is not blank, there was a problem
		if ($status != '')
		{
			flashMsg('warning',$status);
			return FALSE;  //bad, conflicts
		}

		return TRUE;  //good, no conflicts
	}

	// -------------------------------------------------------------------------
	/**
	 * Check that event does not have a conflict before scheduling
	 *
	 * @access    private
	 * @param     string  The string to check.
	 * @return    bool
	 */
	function _check_part_conflict($str)
	{
		//return TRUE;  //turn off participant checking

		// Set the error message to use
		$this->validation->set_message('_check_part_conflict','The %s field cannot be scheduled due to a conflict.');

		// if adding a new event, there can be no conflicts as no location or participant is assigned
		$id = $this->input->post('id', TRUE);
		if ($id == '')
			return TRUE;  //good, no conflicts

		$str = $this->dates_array[$str];  //get date string

		// if date is blank, there can be no conflicts
		if ($str == '0000-00-00 00:00:00' OR $str == '')
			return TRUE;  //good, no conflicts

		// get current participants for this event
		$this->load->model('eventparticipants_model');
		$query = $this->eventparticipants_model->fetchEventParticipants('event.id = '.$id);
		if ($query->num_rows() == 0)
			return TRUE;  //good, no conflicts
		// convert participants query to array
		foreach ($query->result_array() as $row)
		{
			$partsort[$row['participant_sortedname']] = $row['participant_id'];
		}
		$query->free_result();

		// build grid using current participants
		$partGrid = $this->events_model->build_part_grid($partsort);  // Build the participant grid for conflict checking


		// get event data
		$sdttm = strtotime($str);
		$edttm = strtotime("+".($this->input->post('eventLength', TRUE)*3600).' seconds', $sdttm);
		$setup = strtotime("-".($this->input->post('setupTime', TRUE)*3600).' seconds', $sdttm);
		$teardown = strtotime("+".($this->input->post('teardownTime', TRUE)*3600).' seconds', $edttm);
		$eventInfo = array('id' => $id,
							'sdttm' => date('Y-m-d H:i:s',$sdttm),
							'edttm' => date('Y-m-d H:i:s',$edttm),
							'setupdttm' => date('Y-m-d H:i:s',$setup),
							'teardowndttm' => date('Y-m-d H:i:s',$teardown),
							'setup' => $this->input->post('setupTime', TRUE),
							'teardown' => $this->input->post('teardownTime', TRUE),
							'eventlength' => $this->input->post('eventLength', TRUE)
							);

		// check that an event can fit on grid and return status
		$status = check_event_on_partgrid($partGrid, $partsort, $eventInfo, $this->ccsettings->item('interval_length'));

		// if status is not blank, there was a problem
		if ($status != '')
		{
			flashMsg('warning',$status);
			return FALSE;  //bad, conflicts
		}

		return TRUE;  //good, no conflicts
	}


	// --------------------------------------------------------------------------------------------
	// --------------------------------------------------------------------------------------------
	// Reports
	// --------------------------------------------------------------------------------------------
	// --------------------------------------------------------------------------------------------

	// -------------------------------------------------------------------------
	/**
	* Reports Page
	*
	* @access      public
	* @return      none
	*/
	function reports()
	{
		// create urls for use in content view
		$template['base_url']   = 'concentric/events/reports/';
		$template['cancel_url'] = 'concentric/home';

		// Set breadcrumb
		$this->page->set_crumb('Events Reports','concentric/events/reports/');

		// set values for use in content view
		$template['form_attributes'] = array('name' => $this->form_name, 'id' => $this->form_name);
		$template['header'] = $this->form_name;

		// build html for list of events by category
		$template['list_events_by_category'] = $this->_get_list_events_by_category();

		// build content html
		$data['content'] = $this->load->view('/concentric/events/events_reports_menu', $template, TRUE);

		// set values for use in header view
		$data['header']  = $this->form_name;

		// pass all data to container view and display to user
		$this->load->view($this->_container,$data);
	}

	// -------------------------------------------------------------------------
	/**
	 * Build html to call reports by category
	 *
	 * @access    private
	 * @return    html string
	 */
	function _get_list_events_by_category()
	{
		// Load the model
		$this->load->model('categories_model');

		// fetch query object from the model with fields
		$query = $this->categories_model->fetch('categories','id, name','','','name ASC');

		// Loop through to build string
		$data = "\n";
		foreach ($query->result_array() as $row)
		{
			$data .= '<li>' . anchor("concentric/events/publicreports/" . $row['name'],"Report of " . $row['name'] . " Events") . '</li>' . "\n";
		}

		return $data;
	}

	// -------------------------------------------------------------------------
	/**
	* Report all data
	*
	* @access      public
	* @return      none
	*/
	function rep_all($sort = 'alpha')
	{
		// determine sort - default is alphabetically
		if ($sort == 'alpha')
		{
			$sortby = 'event.name ASC';
			$template['sort_method'] = 'Alphabetically';
		} elseif ($sort == 'dttm')      // date/time
		{
			$sortby = 'event.startDateTime ASC, event.name ASC';
			$template['sort_method'] = 'by Date/Time';
		} else
		{
			$sortby = 'event.name ASC';
			$template['sort_method'] = 'Alphabetically';
		}

		$this->load->model('eventlocations_model');    // Instantiate the model
		$this->load->model('eventparticipants_model');    // Instantiate the model

		// fetch query objects from the models
		$template['events_list'] = $this->events_model->fetchEvents('', $sortby);
		//$eventlocations_list = $this->eventlocations_model->fetchEventLocations();
		//$eventparticipants_list = $this->eventparticipants_model->fetchEventParticipants();

		$template['eventlocations_array'] = $this->eventlocations_model->convertLocationsToStr();
		$template['eventparticipants_array'] = $this->eventparticipants_model->convertParticipantsToStr();

		// create urls for use in content view
		$template['base_url']   = 'concentric/events/reports/';
		$template['cancel_url'] = 'concentric/home';

		// Set breadcrumbs
		$this->page->set_crumb('Events Reports','concentric/events/reports/');
		$this->page->set_crumb('List All','concentric/events/rep_all/');

		// set values for use in content view
		$template['form_attributes'] = array('name' => $this->form_name, 'id' => $this->form_name);
		$template['header'] = $this->form_name;
		$template['mode'] = 'internal';

		// build content html using grid view and template data
		$data['content'] = $this->load->view('/concentric/events/events_list', $template, TRUE);

		// set values for use in header view
		$data['header']  = $this->form_name;

		// pass all data to container view and display to user
		$this->load->view($this->_container,$data);
	}

	// -------------------------------------------------------------------------
	/**
	* Download all data
	*
	* @access      public
	* @return      none
	*/
	function rep_download_all($eor = '')
	{
		// Load the helper
		$this->load->helper('download');

		// fetch query object from the model
		$query = $this->events_model->fetchEvents();

		// Initialize values
		$data = '';
		$eor = ($eor == '') ? '' : (',' . $eor);
		$data .= 'id,name,startDateTime,eventLength,setupTime,teardownTime,reimbursibleQ,activeQ,hideQ,hostDescription,hidePanelistsQ,category_name,track_name,locationSetup_name,description' . $eor . "\n";
		// Loop through the data and build the data array
		foreach ($query->result_array() as $row)
		{
			$data .= '"' . $row['id'] . '","' . $row['name'] . '","' . $row['startDateTime'] . '","' . $row['eventLength'] . '","' . $row['setupTime'] . '","' . $row['teardownTime'] . '","' . $row['reimbursibleQ'] . '","' . $row['activeQ'] . '","' . $row['hideQ'] . '","' . $row['hostDescription'] . '","' . $row['hidePanelistsQ'] . '","' . $row['category_name'] . '","' . $row['track_name'] . '","' . $row['locationSetup_name'] . '","' . $row['description'] . '"' . $eor . "\n";
		}

		// Download the file
		force_download('events.csv', $data);

		redirect('/concentric/events/reports', 'location');
	}

	// -------------------------------------------------------------------------
	/**
	* Report grid of scheduled events
	*
	* @access      public
	* @return      none
	*/
	function grid_scheduled_events($mode = 'internal', $timespan = 'default')
	{
		// load grid helper
		$this->load->helper('grid');

		// create urls for use in content view
		$template['base_url']   = 'concentric/events/reports/';

		// Set breadcrumbs
		$this->page->set_crumb('Events Reports','concentric/events/reports/');
		$this->page->set_crumb('Grid of Scheduled Events','concentric/events/rep_scheduled_events/'.$mode);

		// set values for use in content view
		$template['header'] = $this->form_name;

		// figure out interval to use
		// timespan values: default, quarter, half, full
		switch ($timespan) {
			case 'quarter':
				$grid_interval = 900;
				break;
			case 'half':
				$grid_interval = 1800;
				break;
			case 'full':
				$grid_interval = 3600;
				break;
			default:
				$grid_interval = $this->ccsettings->item('interval_length');
				break;
		}

		// setup where clause for locations query
		$where = '';
		if ($mode == 'public')
		{
			$where = array('publicQ'=>'1');
		}

		// Need to create a list of locations for grid
		$this->load->model('locations_model');    // Instantiate the model
		$query = $this->locations_model->fetch('locations', 'id, sortOrder', '', $where, 'sortOrder ASC');  //get data
		foreach ($query->result_array() as $row)
		{
			$locations[$row['sortOrder']] = $row['id'];
		}
		$this->locationGrid = $this->events_model->build_location_grid($locations, $mode, $grid_interval);  // Build the location grid
		$template['content'] = output_locationgrid($this->locationGrid, $locations, $mode);


		// build content html using grid view and template data
		$data['content'] = $this->load->view('/concentric/events/events_reports_grid', $template, TRUE);

		// set values for use in header view
		$data['header']  = $this->form_name;

		// pass all data to container view and display to user
		$this->load->view($this->_container,$data);
	}

	// -------------------------------------------------------------------------
	/**
	* Various Reports for public data
	*
	* @access      public
	* @return      none
	*/
	function publicreports($cat = 'all', $sort = 'alpha', $sched = '')
	{
		// include only public report event items
		$where = 'track.publicQ = 1';
		$where .= ' AND event.hideQ = 0';
		$where .= ' AND event.activeQ = 1';
		if ($sched == 'sched')
		{
			$where .= ' AND event.startDateTime != "0000-00-00 00:00:00"';
		}
		if ($cat != 'all')
		{
			$where .= ' AND category.name = "' . $cat . '"';
		}

		// include only regular panelists
		$where1 = 'participant.equipmentQ = 0';

		// determine sort - default is alphabetically
		if ($sort == 'alpha')
		{
			$sortby = 'event.name ASC';
			$template['sort_method'] = 'Alphabetically';
		} elseif ($sort == 'dttm')
		{
			$sortby = 'event.startDateTime ASC, event.name ASC';
			$template['sort_method'] = 'by Date/Time';
		} else
		{
			$sortby = 'event.name ASC';
			$template['sort_method'] = 'Alphabetically';
		}

		$this->load->model('eventlocations_model');    // Instantiate the model
		$this->load->model('eventparticipants_model');    // Instantiate the model

		// fetch query objects from the models
		$template['events_list'] = $this->events_model->fetchEvents($where, $sortby);
		$eventparticipants_list = $this->eventparticipants_model->fetchEventParticipants($where1);

		$template['eventlocations_array'] = $this->eventlocations_model->convertLocationsToStr();
		$template['eventparticipants_array'] = $this->eventparticipants_model->convertParticipantsToStr($where1, 'participant.sortedName ASC', 'public');

		// create urls for use in content view
		$template['base_url']   = 'concentric/events/reports/';
		$template['cancel_url'] = 'concentric/home';

		// Set breadcrumbs
		$this->page->set_crumb('Events Reports','concentric/events/reports/');
		($cat == 'all') ? $this->page->set_crumb('All Events','concentric/events/publicreports/' . $cat . '/') : $this->page->set_crumb($cat . ' Events','concentric/events/publicreports/' . $cat . '/') ;

		// set values for use in content view
		$template['form_attributes'] = array('name' => $this->form_name, 'id' => $this->form_name);
		$template['header'] = $this->form_name;
		$template['mode'] = 'public';

		// build content html using grid view and template data
		$data['content'] = $this->load->view('/concentric/events/events_list', $template, TRUE);

		// set values for use in header view
		$data['header']  = $this->form_name;

		// pass all data to container view and display to user
		$this->load->view($this->_container,$data);
	}



	// -------------------------------------------------------------------------
	/**
	* Report of events for publication
	*
	* @access      public
	* @return      none
	*/
	function rep_publication($mode = 'publication', $sort = 'category')
	{
		// create urls for use in content view
		$template['base_url']   = 'concentric/events/reports/';
		$template['cancel_url'] = 'concentric/home';

		// Set breadcrumbs
		$this->page->set_crumb('Events Reports','concentric/events/reports/');
		$this->page->set_crumb('Publication Events','concentric/events/rep_publication/');

		// set values for use in content view
		$template['form_attributes'] = array('name' => $this->form_name, 'id' => $this->form_name);
		$template['header'] = $this->form_name;


		// setup where clause for locations query
		$where = '';
		if ($mode == 'public')
		{
			$where = array('publicQ'=>'1');
		}

		// Need to create a list of locations for grid
		$this->load->model('locations_model');    // Instantiate the model
		$query = $this->locations_model->fetch('locations', 'id, sortOrder', '', $where, 'sortOrder ASC');  //get data
		foreach ($query->result_array() as $row)
		{
			$locations[$row['sortOrder']] = $row['id'];
		}
		$template['content'] = $this->events_model->build_event_grid($locations, $mode, $sort);  // Build the location grid


		// build content html using grid view and template data
		$data['content'] = $this->load->view('/concentric/events/events_publication', $template, TRUE);

		// set values for use in header view
		$data['header']  = $this->form_name;

		// pass all data to container view and display to user
		$this->load->view($this->_container,$data);
	}


}


/* End of file events.php */
/* Location: ./system/application/controllers/concentric/events.php */