<?php if (!defined('BASEPATH')) exit('No direct script access allowed');
/**
 * BackendPro
 *
 * A website backend system for developers for PHP 4.3.2 or newer
 *
 * @package     BackendPro
 * @author		Adam Price
 * @copyright	Copyright (c) 2008
 * @license		http://www.gnu.org/licenses/lgpl.html
 * @tutorial	BackendPro.pkg
 */

 /**
 * MODULE NAME   : base_model.php
 *
 * DESCRIPTION   : base model - modified for ConCentric
 *
 * MODIFICATION HISTORY
 *   V1.0   2008-11-11 13:00 PM   - Leane Verhulst     - Created
 *
 * @package			ConCentric
 * @subpackage		base model Class
 * @author			Leane Verhulst
 * @copyright		Copyright (c) 2008
 * @license			http://www.gnu.org/licenses/gpl.html
 */

 // ---------------------------------------------------------------------------

/**
 * Base_model
 *
 * Sets up basic model functions. All user created model classes should
 * extend this to gain access to its basic database model functions.
 *
 * @package			BackendPro
 * @subpackage		Models
 */
class Base_model extends Model
{
	/**
	 * Constructor
	 */
	function Base_model()
	{
		// Inherit from parent class
		parent::Model();

		// Create empty function array
		$this->_TABLES = array();

		log_message('debug','Base_model Class Initialized');
	}

	/**
	 * Fetch
	 *
	 * Fetch table rows from table related to $name. Check no custom
	 * fetch method exists before hand.
	 *
	 * @access public
	 * @param string $name Table Name
	 * @param mixed $fields Fields to return from table
	 * @param array $limit Rows to limit search to
	 * @param mixed $where Return rows that match this
	 * @param mixed $orderby Order to return rows in
	 * @return Query Object
	 */
	function fetch($name, $fields=NULL, $limit=NULL, $where=NULL, $orderby=NULL)
	{
		$func = '_fetch_'.$name;
		if(method_exists($this,$func))
		{
			// There is an overide function
			return call_user_func_array(array($this,$func), array($fields,$limit,$where,$orderby));
		}
		else
		{
			// No override function, procede with fetch
			($fields!=NULL) ? $this->db->select($fields) : '';
			($where!=NULL) ? $this->db->where($where) : '';
			($limit!=NULL) ? $this->db->limit($limit['rows'],$limit['offset']) : '';
			($orderby!=NULL) ? $this->db->order_by($orderby) : '';

			return $this->db->get($this->_TABLES[$name]);
		}
	}

	/**
	 * Insert
	 *
	 * Insert new table data into table related to by $name
	 * Check no custom insert method exists before hand.
	 *
	 * @access public
	 * @param string $name Table Name
	 * @param array $data Data to insert
	 * @return Query Object
	 */
	function insert($name, $data)
	{
		$func = '_insert_' . $name;
		if(method_exists($this,$func))
		{
			// There is an overide function
			return call_user_func_array(array($this,$func), array($data));   
		}
		else
		{
			// No override function, procede with insert
			return $this->db->insert($this->_TABLES[$name],$data);
		}
	}

	/**
	 * addRow - Adds an entry to table and returns either the id
	 *          or false. Uses the base_model function of insert.
	 *
	 * @access	public
	 * @params	string	table name
	 * @params	array	list of fields
	 * @return	mixed	Either id of new entry or false if error
	 */
	function addRow($name, $data)
	{
		$func = '_addRow_' . $name;
		if(method_exists($this, $func))
		{
			// There is an overide function
			return call_user_func_array(array($this,$func), array($data));
		}
		else
		{
			if ($this->insert($name, $data))
			{
				return $this->db->insert_id();
			}
			else
			{
				return FALSE;
			}
		}
	}

	/**
	 * Update
	 *
	 * Update data in table related to by $name
	 * Check no custom update method exists before hand.
	 *
	 * @access public
	 * @param string $name Table Name
	 * @param array $values Data to change
	 * @param mixed $where Rows to update
	 * @return Query Object
	 */
	function update($name, $values, $where)
	{
		$func = '_update_' . $name;
		if(method_exists($this,$func))
		{
			// There is an overide function
			return call_user_func_array(array($this,$func), array($values,$where));  
		}
		else
		{
			// No overside function, procede with general update
			$this->db->where($where);
			return $this->db->update($this->_TABLES[$name],$values);
		}
	}

	/**
	 * Delete
	 *
	 * Delete rows from table related to by $name
	 * Check no custom delete method exists before hand.
	 *
	 * @access public
	 * @param string $name Table Name
	 * @param mixed $where Rows to delete
	 * @return Query Object
	 */
	function delete($name, $where)
	{
		$func = '_delete_' . $name;
		if(method_exists($this, $func))
		{
			// There is an overide function
			return call_user_func_array(array($this,$func), array($where));    
		}
		else
		{
			// No overside function, procede with general delete
			$this->db->where($where);
			return $this->db->delete($this->_TABLES[$name]);
		}
	}

	/**
	 * Count_condition
	 *
	 * Returns a count of the number of rows in a table that matches a condition.
	 * Check no custom count condition method exists before hand.
	 *
	 * @access public
	 * @param  string   $table Table Name
	 * @param  mixed    $where Condition to match
	 * @return integer  Count of the rows
	 */
	function count_condition($table, $where = '')
	{
		if ($where == '')
			return FALSE;

		$func = '_count_condition_' . $table;
		if(method_exists($this, $func))
		{
			// There is an overide function
			return call_user_func_array(array($this, $func), array($where));    
		}
		else
		{
			$table = $this->config->item('concentric_table_prefix') . $table;

			$this->db->select('COUNT(*) cnt');
			$this->db->from($table);
			$this->db->where($where);
			$query = $this->db->get();
			$row = $query->row();
			$query->free_result();
			return $row->cnt;
		}
	}

}


/* End of file base_model.php */
/* Location: ./system/application/models/base_model.php */
